#!/usr/bin/python
################################################################################
## @file
##
## @copyright
## Copyright (c) 2016 Intel Corporation. All rights reserved
## This software and associated documentation (if any) is furnished
## under a license and may only be used or copied in accordance
## with the terms of the license. Except as permitted by the
## license, no part of this software or documentation may be
## reproduced, stored in a retrieval system, or transmitted in any
## form or by any means without the express written consent of
## Intel Corporation.
## This file contains a 'Sample Driver' and is licensed as such
## under the terms of your license agreement with Intel or your
## vendor.  This file may be modified by the user, subject to
## the additional terms of the license agreement
################################################################################

import os
import re
import sys

PirqFileNameMatch = re.compile("\w+_PirqTableData.h")
PirqDev = re.compile("^\s*{\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+)\s*}\s*,(.*)$")

IrqApicFileNameMatch = re.compile("\w+_MpTableData.h")
IrqApicDev = re.compile("^\s*{\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+)\s*}\s*,(.*)$")
ApicData = re.compile("^\s*{\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+)\s*}\s*,(.*)$")

GuidData = re.compile("^\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+),\s*(0[xX][0-9a-fA-F]+|\d+)\s*\\\s*$")

def FixAceOutput(BuildDir):
    FixAcePirqTables(BuildDir)
    FixAceIrqApicTables(BuildDir)
    FixPlatformOpRomTables(BuildDir)

def FixAcePirqTables(BuildDir):
    for dirName, subdirList, fileList in os.walk(BuildDir):
        for fileName in fileList:
            if PirqFileNameMatch.match(fileName) is not None:
                FixPirqFile("%s%s%s"%(dirName, os.sep, fileName))

def FixPirqFile(FilePath):
    newFile = []
    f = open(FilePath, 'r')
    line = f.readline()
    while line != "":
        data = PirqDev.match(line.rstrip())
        if data:
            PciBusNum = data.group(1)
            PciDevNum = data.group(2)
            IntAReg = data.group(3)
            IntAPin = data.group(4)
            IntBReg = data.group(5)
            IntBPin = data.group(6)
            IntCReg = data.group(7)
            IntCPin = data.group(8)
            IntDReg = data.group(9)
            IntDPin = data.group(10)
            SlotNum = data.group(11)
            Reserved = data.group(12)
            Comment = data.group(13)
            newLine = "  { %s,    %s,  { { %s,     %s },       { %s,     %s },       { %s,     %s },       { %s,     %s } },         %s,       %s },%s\n"%(PciBusNum, PciDevNum, IntAReg, IntAPin, IntBReg, IntBPin, IntCReg, IntCPin, IntDReg, IntDPin, SlotNum, Reserved, Comment)
            newFile.append(newLine)
        else:
            newFile.append(line)
        line = f.readline()
    f.close()
    f = open(FilePath, 'w')
    f.writelines(newFile)
    f.close()

def FixAceIrqApicTables(BuildDir):
    for dirName, subdirList, fileList in os.walk(BuildDir):
        for fileName in fileList:
            if IrqApicFileNameMatch.match(fileName) is not None:
                FixIrqApicFile("%s%s%s"%(dirName, os.sep, fileName))

def FixIrqApicFile(FilePath):
    newFile = []
    f = open(FilePath, 'r')
    line = f.readline()
    while line != "":
        data = IrqApicDev.match(line.rstrip())
        if data:
            PciBusNum = data.group(1)
            PciDevNum = data.group(2)
            IntAIn = data.group(3)
            IntAId = data.group(4)
            IntBIn = data.group(5)
            IntBId = data.group(6)
            IntCIn = data.group(7)
            IntCId = data.group(8)
            IntDIn = data.group(9)
            IntDId = data.group(10)
            Reserved = data.group(11)
            Comment = data.group(12)
            newLine = "  { %s,    %s,   { {%s,     %s},  {%s,     %s},  {%s,     %s},  {%s,     %s} }, %s },%s\n"%(PciBusNum, PciDevNum, IntAIn, IntAId, IntBIn, IntBId, IntCIn, IntCId, IntDIn, IntDId, Reserved, Comment)
            newFile.append(newLine)
        apic = ApicData.match(line.rstrip())
        if apic:
            ApicAddress = apic.group(1)
            ApicBuild = apic.group(2)
            ApicRun = apic.group(3)
            Comment = apic.group(4)
            newLine = "  {{%s},%s,      %s },%s\n"%(ApicAddress, ApicBuild, ApicRun, Comment)
            newFile.append(newLine)
        if data is None and apic is None:
            newFile.append(line)
        line = f.readline()
    f.close()
    f = open(FilePath, 'w')
    f.writelines(newFile)
    f.close()

def FixPlatformOpRomTables(BuildDir):
    for dirName, subdirList, fileList in os.walk(BuildDir):
        for fileName in fileList:
            if fileName == "PlatformOpRomTable.h":
                FixPlatformOpRomFile("%s%s%s"%(dirName, os.sep, fileName))

def FixPlatformOpRomFile(FilePath):
    newFile = []
    f = open(FilePath, 'r')
    line = f.readline()
    while line != "":
        data = GuidData.match(line.rstrip())
        if data:
            num1 = data.group(1)
            num2 = data.group(2)
            num3 = data.group(3)
            num4 = data.group(4)
            num5 = data.group(5)
            num6 = data.group(6)
            num7 = data.group(7)
            num8 = data.group(8)
            num9 = data.group(9)
            num10 = data.group(10)
            num11 = data.group(11)
            newLine = "    %s, %s, %s, { %s, %s, %s, %s, %s, %s, %s, %s } \\\n"%(num1, num2, num3, num4, num5, num6, num7, num8, num9, num10, num11)
            newFile.append(newLine)
        else:
            newFile.append(line)
        line = f.readline()
    f.close()
    f = open(FilePath, 'w')
    f.writelines(newFile)
    f.close()

def main():
    if len(sys.argv) <= 1:
        print "Usage: %s <Build-Directory>"%sys.argv[0]
    else:
        FixAceOutput(sys.argv[1])

if __name__ == "__main__":
    main()
